using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;
using Microsoft.Xna.Framework.Net;
using Microsoft.Xna.Framework.Storage;

using XNACS1Lib;

namespace ClassExample
{
    /// <summary>
    /// This is the main type for your game
    /// </summary>
    public class ClassExample : XNACS1Base
    {
        const float kWorldWidth = 200f;
        const float kTargetMinSpeed = 0.001f * kWorldWidth;
        const float kTargetMaxSpeed = 0.005f * kWorldWidth;
        const float kTargetMinRadius = 0.01f * kWorldWidth;
        const float kTargetMaxRadius = 0.03f * kWorldWidth;

        CircleWithMass mTheTarget;
        CircleWithMass mTheHero;

        XNACS1Circle mCollidPt;

        float mCollideTime;

        protected override void InitializeWorld()
        {
            World.SetWorldCoordinate(new Vector2(0f, 0f), kWorldWidth);

            mTheTarget = new CircleWithMass();
            mTheTarget.ShouldTravel = true;
            mTheTarget.Color = Color.Red;

            mTheHero = new CircleWithMass();
            mTheHero.Color = Color.Green;
            RandomizeTarget();

            mCollidPt = new XNACS1Circle();
            mCollidPt.Color = Color.Black;
            mCollidPt.RemoveFromAutoDrawSet();

            mCollideTime = 0f;
        }

        protected override void UpdateWorld()
        {
            if (GamePad.ButtonBackClicked())
                Exit();

            if (mCollideTime > 0)
                mCollideTime--;

            if ((GamePad.ThumbSticks.Right.X != 0) || (GamePad.ThumbSticks.Right.Y != 0))
            {
                mTheHero.ShouldTravel = false;
                mTheHero.Radius += 0.1f * GamePad.ThumbSticks.Right.X;
                mTheHero.Speed += 0.1f * GamePad.ThumbSticks.Right.Y;
                
                mCollidPt.RemoveFromAutoDrawSet();
            }

            mTheHero.Update(mTheTarget);
            mTheTarget.Update(null);

            if (GamePad.ButtonBClicked())
            {
                RandomizeTarget();
                mTheHero.ShouldTravel = false;
                mCollidPt.RemoveFromAutoDrawSet();
            }

            if (GamePad.ButtonAClicked())
            {
                mCollideTime = ComputeHeroVDirection();
                if (mCollideTime > 0)
                    mTheHero.ShouldTravel = true;
            }

            float heroP = mTheHero.Mass() * mTheHero.Velocity.LengthSquared();
            float targetP = mTheTarget.Mass() * mTheTarget.Velocity.LengthSquared() ;
            EchoToTopStatus("Energy=" + (heroP + targetP) + "(Hero=" + heroP + " Target="+targetP+ ")   Colliding in: " + mCollideTime);
            EchoToBottomStatus("Mass: Hero=" + mTheHero.Mass() + " Target=" + mTheTarget.Mass() + "    Speed: Hero=" + mTheHero.Speed + " Target="+mTheTarget.Speed);
        }


        // 
        // we want:
        //      hv - hero velocity (hvx, hvy)
        //
        // given:
        //      hs - hero speed
        //      hp - current hero position
        //      tv - target velocity (tvx, tvy)
        //      tp - current target position
        // 
        // we know:  hvx*hvx + hvy*hvy = hs*hs
        //
        // we want to find tm, hvx, hvy such that:
        //
        //    tpx + tm * tvx = hpx + tm * hvx
        //    tpy + tm * tvy = hpy + tm * hvy
        //    hvx * hvx + hvy * hvy = hs * hs
        //
        private float ComputeHeroVDirection()
        {
            Vector2 dist = mTheTarget.Center - mTheHero.Center;
            // a * mt * mt + b * mt + c = 0
            float a = mTheTarget.Velocity.LengthSquared() - (mTheHero.Speed * mTheHero.Speed);
            float b = 2f * Vector2.Dot(mTheTarget.Velocity, dist);
            float c = dist.LengthSquared();

            mCollidPt.RemoveFromAutoDrawSet();
            float tm = 0f;

            float dm = (b * b) - (4 * a * c);
            if (dm > 0f)
            {
                dm = (float)(Math.Sqrt(dm));
                float t1 = (-b + dm) / (2f * a);
                float t2 = (-b - dm) / (2f * a);
                if (t1 > 0)
                {
                    if (t2 > 0)
                    {
                        tm = (t1 < t2) ? t1 : t2;
                    }
                    else
                    {
                        tm = t1;
                    }
                }
                else
                {
                    if (t2 > 0)
                        tm = t2;
                }
                if (tm > 0)
                {
                    Vector2 collidPt = mTheTarget.Center + tm * mTheTarget.Velocity;
                    Vector2 toHero = collidPt - mTheHero.Center;
                    mTheHero.Velocity = (1f / tm) * toHero;

                    mCollidPt.Center = collidPt;
                    mCollidPt.AddToAutoDrawSet();
                }
            }
            return tm;
        }

        private void RandomizeTarget()
        {
            mTheTarget.CenterX = World.WorldDimension.X / 2f;
            mTheTarget.CenterY = World.WorldDimension.Y / 2f;
            mTheTarget.VelocityDirection = new Vector2(RandomFloat(-1f, 1f), RandomFloat(-1f, 1f));
            mTheTarget.Speed = RandomFloat(kTargetMinSpeed, kTargetMaxSpeed);

            mTheTarget.Radius = RandomFloat(kTargetMinRadius, kTargetMaxRadius);
        }
    }
}
