using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;
using Microsoft.Xna.Framework.Net;
using Microsoft.Xna.Framework.Storage;

using XNACS1Lib;

namespace ClassExample
{
    /// <summarmy>
    /// A Fake class, controls what appears in application window using
    ///         XNACS1Base.World
    /// This is a "fake" class because we cannot instantiate two of these!!
    /// </summarmy>
    public class Camera
    {
        /// Define "safe zone" anyway you like, here I am going to say, "safe zone" is
        /// 90% of the Width/Height. This is a "arbitrary" number I am using for this example.
        private float mSafeZone = 0.9f;

        /// <summary>
        /// Create a "camera" looking at the "center" position and with specified width
        /// Height = width * XNACS1Base.World.WorldDimension.Y/XNACS1Base.World.WorldDimension.X
        /// </summary>
        /// <param name="focusCenter">Center position</param>
        /// <param name="worldWidth">width of what the camera can see</param>
        public Camera(Vector2 focusCenter, float worldWidth)
        {
            Vector2 lowerLeft = new Vector2();
            float worldHeight = worldWidth * XNACS1Base.World.WorldDimension.Y / XNACS1Base.World.WorldDimension.X;

            lowerLeft.X = focusCenter.X - (worldWidth / 2f);
            lowerLeft.Y = focusCenter.Y - (worldHeight / 2f);

            XNACS1Base.World.SetWorldCoordinate(lowerLeft, worldWidth);
        }

        /// <summary>
        /// Move the camera by the input delta
        /// </summary>
        /// <param name="delta">Move by this X and Y</param>
        public void MoveCamera(Vector2 delta)
        {
            XNACS1Base.World.SetWorldCoordinate(
                    XNACS1Base.World.WorldMin + delta,
                    XNACS1Base.World.WorldDimension.X);
        }

        /// <summary>
        /// Change the width of what the camera can see by delta.                   
        /// </summary>
        /// <param name="deltaX">change in width of what the camera can see</param>
        public void ZoomBy(float deltaX)
        {

            float newW = XNACS1Base.World.WorldDimension.X + deltaX;
            float newH = newW * (XNACS1Base.World.WorldDimension.Y/XNACS1Base.World.WorldDimension.X);

            float dx = 0.5f * (newW - XNACS1Base.World.WorldDimension.X);
            float dy = 0.5f * (newH - XNACS1Base.World.WorldDimension.Y);
            Vector2 newOrg = XNACS1Base.World.WorldMin - new Vector2(dx, dy);

            XNACS1Base.World.SetWorldCoordinate(newOrg, newW);

            XNACS1Base.EchoToTopStatus("Center=" +
               ((XNACS1Base.World.WorldMax + XNACS1Base.World.WorldMin) * 0.5f)
               + "  Dimension:" + XNACS1Base.World.WorldDimension);
        }

        /// <summary>
        /// Gets/Sets safe Zone percentage. Value should be between 0 to 0.5. 
        /// Safe zone of greater than 0.5 forces oscillation of the camera!
        /// </summary>
        public float SafeZonePercentage
        {
            get { return mSafeZone; }
            set
            {
                mSafeZone = value;
                if (mSafeZone > 0.45f)
                    mSafeZone = 0.45f;
                else if (mSafeZone < 0f)
                    mSafeZone = 0f;
            }
        }


        /// <summary>
        /// Determines if pt is within the safeZone of the camera, if not, pan the camrea to 
        /// make sure pt is within the safeZone
        /// </summary>
        /// <param name="pt"></param>
        public void KeepPointInSafeZone(Vector2 pt)
        {
            Vector2 safeV = (1f - SafeZonePercentage) * XNACS1Base.World.WorldDimension;
            Vector2 delta = Vector2.Zero;

            // test for left and right
            float rightMove = pt.X - (XNACS1Base.World.WorldMax.X - safeV.X);
            if (rightMove > 0f)
            {
                delta.X = rightMove;
            }
            else
            {
                float leftMove = (XNACS1Base.World.WorldMin.X + safeV.X) - pt.X;
                if (leftMove > 0f)
                    delta.X = -leftMove;
            }

            // now up and down
            float upMove = pt.Y - (XNACS1Base.World.WorldMax.Y - safeV.Y);
            if (upMove > 0f)
            {
                delta.Y = upMove;
            }
            else
            {
                float downMove = (XNACS1Base.World.WorldMin.Y + safeV.Y) - pt.Y;
                if (downMove > 0f)
                    delta.Y = -downMove;
            }
            MoveCamera(delta);
        }
    }
}
