﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Animation;
using System.Windows.Shapes;
using Microsoft.Phone.Controls;


// for BitmapImage
using System.Windows.Media.Imaging;

// for StreamResourceInfo
using System.Windows.Resources;

namespace TouchManipulation
{
    public partial class MainPage : PhoneApplicationPage
    {
        ImageBrush mImageToShow;

        // Constructor
        public MainPage()
        {
            InitializeComponent();
            
            BitmapImage img = new BitmapImage();
            Uri uri = new Uri("Skater.png", UriKind.Relative);
            StreamResourceInfo resourceInfo = Application.GetResourceStream(uri);
            img.SetSource(resourceInfo.Stream);

            mImageToShow = new ImageBrush();
            mImageToShow.ImageSource = img;
            mImageToShow.Stretch = Stretch.UniformToFill;
            mImageToShow.Transform = new CompositeTransform();

            mImageRectangle.Fill = mImageToShow;
        }

        #region TextBox service routines
        /// <summary>
        /// "Get Focus" service routine: good hint that user wants to enter something.
        /// Here we simply select all the text, user can choose to erase everything.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void NewTextString(object sender, RoutedEventArgs e)
        {
            mInputBox.SelectAll();
        }

        /// <summary>
        /// "Lost Focus" Service routine: when text input is done will lost focus.
        /// This is a good hint we have a complete string. In this case, verify the input
        /// by echoing it in the label area.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void TextEntered(object sender, RoutedEventArgs e)
        {
            mLabel.Text = mInputBox.Text;
        }
        #endregion 

        private void LoadImage(object sender, RoutedEventArgs e)
        {
            BitmapImage img = new BitmapImage();
            Uri uri = new Uri(mInputBox.Text, UriKind.Relative);
            StreamResourceInfo resourceInfo = Application.GetResourceStream(uri);
            if (resourceInfo != null)
            {
                img.SetSource(resourceInfo.Stream);
                mImageToShow.ImageSource = img;
            }
            else
                mLabel.Text += ": Image not found";
        }

        #region Manipulation on the image    
        
        private void mImageCanvas_ManipulationStarted(object sender, ManipulationStartedEventArgs e)
        {
            mLabel.Text = "Started";
        }

        private void mImageCanvas_ManipulationDelta(object sender, ManipulationDeltaEventArgs e)
        {
            ManipulationDelta delta = e.DeltaManipulation;
            // delta = e.CumulativeManipulation;
            CompositeTransform t = mImageToShow.Transform as CompositeTransform;
            t.TranslateX += delta.Translation.X;
            t.TranslateY += delta.Translation.Y;
            mLabel.Text = "Translation:" + delta.Translation;
        }

        private void mImageCanvas_ManipulationCompleted(object sender, ManipulationCompletedEventArgs e)
        {
            mLabel.Text = "Completed";
        }
        #endregion

        #region toolkit gesture event service support
        /// <summary>
        /// There are at least three places posting the following solution, I referenced (copied) from what seem
        /// like the first posting ...
        /// http://www.frenk.com/2011/03/windows-phone-7-correct-pinch-zoom-in-silverlight/ (Mar 2 2011?)
        /// http://mobile.dzone.com/articles/correcting-pinch-zoom (Feb 8 2012)
        /// http://www.codeproject.com/Articles/335069/Nice-panning-and-zooming-in-Windows-Phone-7 (Feb24 2012)
        /// </summary>
        double mTotalImageScale = 1;
        Point mTranslationRef = new Point(0, 0);
        
        double mInitScaleFactor;
        Point mInitFinger1, mInitFinger2;

        private void GestureListener_PinchStarted(object sender, PinchStartedGestureEventArgs e)
        {
            mInitFinger1 = e.GetPosition(mImageRectangle, 0);
            mInitFinger2 = e.GetPosition(mImageRectangle, 1);
            mInitScaleFactor = 1;
        }

        private void GestureListener_PinchDelta(object sender, PinchGestureEventArgs e)
        {
            double scaleFactor = e.DistanceRatio / mInitScaleFactor;
            Point currentFinger1 = e.GetPosition(mImageRectangle, 0);
            Point currentFinger2 = e.GetPosition(mImageRectangle, 1);

            Point translationDelta = GetTranslationDelta(
                    currentFinger1,
                    currentFinger2,
                    mInitFinger1,
                    mInitFinger2,
                    mTranslationRef,
                    scaleFactor);

            mInitFinger1 = currentFinger1;
            mInitFinger2 = currentFinger2;
            mInitScaleFactor = e.DistanceRatio;

            UpdateImage(scaleFactor, translationDelta);
        }


        private void UpdateImage(double scaleFactor, Point delta)
        {
            mTotalImageScale *= scaleFactor;
            mTranslationRef = new Point(mTranslationRef.X + delta.X, mTranslationRef.Y + delta.Y);

            CompositeTransform t = mImageToShow.Transform as CompositeTransform;
            t.ScaleX = mTotalImageScale;
            t.ScaleY = mTotalImageScale;
            t.TranslateX = mTranslationRef.X;
            t.TranslateY = mTranslationRef.Y;
        }

        private Point GetTranslationDelta(
            Point currentFinger1, Point currentFinger2,
            Point oldFinger1, Point oldFinger2,
            Point currentPosition, double scaleFactor)
        {
            var newPos1 = new Point(
                currentFinger1.X + (currentPosition.X - oldFinger1.X) * scaleFactor,
                currentFinger1.Y + (currentPosition.Y - oldFinger1.Y) * scaleFactor);

            var newPos2 = new Point(
                currentFinger2.X + (currentPosition.X - oldFinger2.X) * scaleFactor,
                currentFinger2.Y + (currentPosition.Y - oldFinger2.Y) * scaleFactor);

            var newPos = new Point(
                (newPos1.X + newPos2.X) / 2,
                (newPos1.Y + newPos2.Y) / 2);

            return new Point(
                newPos.X - currentPosition.X,
                newPos.Y - currentPosition.Y);
        }

        #endregion
    }
}