﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Animation;
using System.Windows.Shapes;
using Microsoft.Phone.Controls;


/// File: MainPage.xaml.cs
/// Author: Kelvin Sung
/// Purpose: show how to load image within the project
/// 

// for BitmapImage
using System.Windows.Media.Imaging;

// for StreamResourceInfo
using System.Windows.Resources;

namespace PinchManipulate
{
    public partial class MainPage : PhoneApplicationPage
    {
        ImageBrush mImageToShow;

        // Constructor
        public MainPage()
        {
            InitializeComponent();
            
            BitmapImage img = new BitmapImage();
            Uri uri = new Uri("Skater.png", UriKind.Relative);
            StreamResourceInfo resourceInfo = Application.GetResourceStream(uri);
            img.SetSource(resourceInfo.Stream);

            mImageToShow = new ImageBrush();
            mImageToShow.ImageSource = img;
            mImageToShow.Stretch = Stretch.UniformToFill;
            mImageToShow.Transform = new CompositeTransform();

            mImageRectangle.Fill = mImageToShow;
        }

        #region TextBox service routines
        /// <summary>
        /// "Get Focus" service routine: good hint that user wants to enter something.
        /// Here we simply select all the text, user can choose to erase everything.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void NewTextString(object sender, RoutedEventArgs e)
        {
            mInputBox.SelectAll();
        }

        /// <summary>
        /// "Lost Focus" Service routine: when text input is done will lost focus.
        /// This is a good hint we have a complete string. In this case, verify the input
        /// by echoing it in the label area.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void TextEntered(object sender, RoutedEventArgs e)
        {
            mLabel.Text = mInputBox.Text;
        }
        #endregion 

        private void LoadImage(object sender, RoutedEventArgs e)
        {
            BitmapImage img = new BitmapImage();
            Uri uri = new Uri(mInputBox.Text, UriKind.Relative);
            StreamResourceInfo resourceInfo = Application.GetResourceStream(uri);
            if (resourceInfo != null)
            {
                img.SetSource(resourceInfo.Stream);
                mImageToShow.ImageSource = img;
            }
            else
                mLabel.Text += ": Image not found";
        }

        #region toolkit gesture event service support
        /// <summary>
        /// There are at least three places posting the following solution, I referenced from what seem
        /// like the first posting ...
        ///     http://www.frenk.com/2011/03/windows-phone-7-correct-pinch-zoom-in-silverlight/ (Mar 2 2011?)
        /// here are two other identical postings (neither reference to the first one, 
        /// makes one wonder just who copied the code from whom, and when, CITE YOUR SOURCES!!):
        ///     http://mobile.dzone.com/articles/correcting-pinch-zoom (Feb 8 2012)
        ///     http://www.codeproject.com/Articles/335069/Nice-panning-and-zooming-in-Windows-Phone-7 (Feb24 2012)
        /// All three solutions implemented intraction based on intermediate results (potentially inaccurate), and
        /// did not bother to optimize their solution. I changed the implementaiton to always reference initial values
        /// and optimized the runtime GetTranslateData() into initializations in the PinchStart (once instead of continuous
        /// calling during delta).
        /// </summary>
        Point mTranslationRef = new Point(0, 0);
        double mScaleRef = 1f;
        Point mDeltaVec = new Point(0, 0); 
                // we only need to keep track of translation 
                // and for translation, we only care about the middle
                // position of the two fingers

        // OK, so the correct translation/scale we want to perform is:
        //
        //    1. move the initial middle of the two finger back to the initial translation
        //    2. scale up by the current scaling
        //    3. move back to the current middle of the two fingers
        //  
        // If, 
        //       if1, if2: are the initial finger touch
        //       mTranslationRef:  is the initial translation
        //       s: is the current scaling factor [this is e.DistanceRatio]
        //              e.DistanceRaio is the currentDistanceBetweenFingers/initDistanceBetweenFingers
        //       cf1, cf1: is the current finger position
        // 
        // what we want is:
        //
        //       im = 0.5 * (if1 + if2)             <-- middle position of the two initial finger position
        //       mDeltaVec = mTranslationRef - im   <-- translate back by im
        //       cm = 0.5 * (cf1 + cf2)             <-- middle position of the two current finger position
        // 
        // Translation:  cm + (mDeltaVec * s)
        //   

        private void GestureListener_PinchStarted(object sender, PinchStartedGestureEventArgs e)
        {
            Point if1 = e.GetPosition(mImageRectangle, 0);      // position of first finger
            Point if2 = e.GetPosition(mImageRectangle, 1);      // position of second finger
            Point im = new Point((0.5f * (if1.X + if2.X)),      // middle between the two fingers
                                 (0.5f * (if1.Y + if2.Y)));
            
            CompositeTransform t = mImageToShow.Transform as CompositeTransform;
            mTranslationRef.X = t.TranslateX;
            mTranslationRef.Y = t.TranslateY;
            mScaleRef = t.ScaleX;  // X and Y scales are always the same in our case

            // mDeltaVec will take the middle of initial touch position to the 
            // initial translation position. This will give the proper effect of 
            // scaling with respect to the initial touch position
            mDeltaVec.X = mTranslationRef.X - im.X;
            mDeltaVec.Y = mTranslationRef.Y - im.Y;

            mLabel.Text = "Pinch started:" + mTranslationRef;
        }

        private void GestureListener_PinchDelta(object sender, PinchGestureEventArgs e)
        {
            Point cf1 = e.GetPosition(mImageRectangle, 0); // current finger 1
            Point cf2 = e.GetPosition(mImageRectangle, 1); // current finger 2

            Point cm = new Point((0.5f * (cf1.X + cf2.X)), (0.5f * (cf1.Y + cf2.Y)));

            Point newPos = new Point(cm.X + (mDeltaVec.X * e.DistanceRatio),
                                     cm.Y + (mDeltaVec.Y * e.DistanceRatio));

            CompositeTransform t = mImageToShow.Transform as CompositeTransform;
            t.ScaleX = e.DistanceRatio * mScaleRef;
            t.ScaleY = e.DistanceRatio * mScaleRef;
            t.TranslateX = newPos.X;
            t.TranslateY = newPos.Y;

            mLabel.Text = string.Format("Delta:s={0:F2} ({1:F2},{2:F2})", e.DistanceRatio, newPos.X, newPos.Y); ;
        }

        private void GestureListener_PinchCompleted(object sender, PinchGestureEventArgs e)
        {

        }

        #endregion
    }
}